(() => {
  const ENTRY_ID = 'uom-upload-entry';
  const BUTTON_CLASS = 'uom-upload-entry__button';
  const UOM_URL = 'https://api.uom.cn/?mod=cc&act=upload_page';
  const WINDOW_NAME = 'uomUploadWindow';
  let teardownPositioning = null;
  let targetElement = null;

  const waitForBody = () =>
    new Promise((resolve) => {
      if (document.body) {
        resolve(document.body);
        return;
      }
      const observer = new MutationObserver(() => {
        if (document.body) {
          observer.disconnect();
          resolve(document.body);
        }
      });
      observer.observe(document.documentElement, { childList: true, subtree: true });
    });

  const getViewportScreenOffset = () => {
    const verticalChrome = Math.max(0, window.outerHeight - window.innerHeight);
    return {
      left: window.screenX,
      top: window.screenY + verticalChrome
    };
  };

  const DEFAULT_WIDTH = 460;
  const DEFAULT_HEIGHT = 640;

  const calculatePopupPosition = () => {
    const margin = 16;
    const horizontalGap = 0;
    const viewportWidth = window.innerWidth || DEFAULT_WIDTH;
    const viewportHeight = window.innerHeight || DEFAULT_HEIGHT;
    const viewportOffset = getViewportScreenOffset();
    const viewportLeft = viewportOffset.left;
    const viewportTop = viewportOffset.top;
    const viewportRight = viewportLeft + viewportWidth;
    const viewportBottom = viewportTop + viewportHeight;

    const entry = document.getElementById(ENTRY_ID);
    const button = entry?.querySelector(`.${BUTTON_CLASS}`);
    const buttonRect = button?.getBoundingClientRect();
    const targetRect = targetElement?.getBoundingClientRect();

    let left;
    if (buttonRect) {
      left = viewportLeft + buttonRect.left;
    } else if (targetRect) {
      left = viewportLeft + targetRect.right + horizontalGap;
    } else {
      left = viewportRight - DEFAULT_WIDTH - margin;
    }

    const minLeft = viewportLeft + margin;
    left = Math.max(left, minLeft);

    let top;
    let height;
    if (buttonRect) {
      top = viewportTop + buttonRect.bottom;
      height = viewportBottom - top;
      if (height <= 0) {
        height = Math.min(DEFAULT_HEIGHT, viewportHeight - margin * 2);
        top = Math.max(viewportTop + margin, viewportBottom - height - margin);
      }
    } else if (targetRect) {
      top = viewportTop + targetRect.bottom + margin;
      height = Math.min(DEFAULT_HEIGHT, viewportBottom - top - margin);
      const maxTop = viewportBottom - height - margin;
      top = Math.min(Math.max(top, viewportTop + margin), maxTop);
    } else {
      top = viewportTop + margin;
      height = Math.min(DEFAULT_HEIGHT, viewportHeight - margin * 2);
    }

    if (!Number.isFinite(height) || height <= 0) {
      height = DEFAULT_HEIGHT;
      top = Math.max(viewportTop + margin, viewportBottom - height - margin);
    }

    let width;
    if (buttonRect || targetRect) {
      width = viewportRight - left;
    } else {
      width = Math.min(DEFAULT_WIDTH, viewportRight - left - margin);
    }

    if (width <= 0 || Number.isNaN(width)) {
      width = DEFAULT_WIDTH;
      left = Math.max(viewportLeft + margin, viewportRight - width - margin);
    }

    return {
      width,
      height,
      left: Math.round(left),
      top: Math.round(top)
    };
  };

  const openWindowDirectly = (bounds) => {
    const { width, height, left, top } = bounds || calculatePopupPosition();
    const features = [
      `width=${width}`,
      `height=${height}`,
      `left=${left}`,
      `top=${top}`,
      'resizable=yes',
      'scrollbars=yes',
      'toolbar=no',
      'location=no',
      'menubar=no'
    ].join(',');
    const handler = window.open(UOM_URL, WINDOW_NAME, features);
    if (!handler || handler.closed) {
      alert('请允许浏览器弹窗以打开 https://api.uom.cn');
    } else {
      handler.focus();
    }
  };

  const requestPopupViaExtension = (bounds) =>
    new Promise((resolve, reject) => {
      if (
        typeof chrome === 'undefined' ||
        !chrome.runtime ||
        typeof chrome.runtime.sendMessage !== 'function'
      ) {
        reject(new Error('chrome.runtime 不可用'));
        return;
      }

      chrome.runtime.sendMessage(
        {
          action: 'openUploadWindow',
          payload: bounds
        },
        (response) => {
          if (chrome.runtime.lastError) {
            reject(new Error(chrome.runtime.lastError.message || '消息发送失败'));
            return;
          }
          if (response?.success) {
            resolve();
            return;
          }
          reject(new Error(response?.error || '未知错误'));
        }
      );
    });

  const openUploadWindow = () => {
    try {
      const bounds = calculatePopupPosition();
      requestPopupViaExtension(bounds).catch((error) => {
        console.warn('[UOM 上传助手] 扩展弹窗失败, 回退到 window.open。', error);
        openWindowDirectly(bounds);
      });
    } catch (error) {
      console.error('[UOM 上传助手] 计算弹窗位置失败:', error);
      alert('无法打开上传窗口, 请检查浏览器弹窗设置或查看控制台日志。');
    }
  };

  const createEntry = () => {
    const entry = document.createElement('div');
    entry.id = ENTRY_ID;
    entry.className = 'uom-upload-entry';

    const button = document.createElement('button');
    button.type = 'button';
    button.className = BUTTON_CLASS;
    button.textContent = '上传文件';
    button.addEventListener('click', openUploadWindow);

    entry.appendChild(button);
    return entry;
  };

  const cleanupPositioning = () => {
    if (typeof teardownPositioning === 'function') {
      teardownPositioning();
      teardownPositioning = null;
    }
  };

  const findTargetDiv = () => {
    // 优先查找新版编辑器的容器，兼容旧版
    return (
      document.querySelector('.editor-v-root') ||
      document.getElementById('js_appmsg_editor') ||
      document.querySelector('.appmsg_editor_wrp')
    );
  };

  const ensureEntryExists = () => {
    if (!document.body) {
      return;
    }

    const targetDiv = findTargetDiv();
    if (!targetDiv) {
      // 如果找不到目标div，延迟重试
      setTimeout(ensureEntryExists, 500);
      return;
    }

    targetElement = targetDiv;

    cleanupPositioning();

    const existingEntry = document.getElementById(ENTRY_ID);
    if (existingEntry) {
      existingEntry.remove();
    }

    const entry = createEntry();
    document.body.appendChild(entry);

    // 使用固定定位，计算按钮位置使其初始紧贴目标 div 右侧
    const updateButtonPosition = () => {
      const rect = targetDiv.getBoundingClientRect();
      const buttonWidth = entry.offsetWidth || 0;
      const buttonHeight = entry.offsetHeight || 0;
      const margin = 16;
      const horizontalGap = 12;

      const left = Math.min(
        Math.max(rect.right + horizontalGap, margin),
        window.innerWidth - buttonWidth - margin
      );
      const top = Math.min(
        Math.max(rect.top, margin),
        window.innerHeight - buttonHeight - margin
      );

      entry.style.position = 'fixed';
      entry.style.left = `${left}px`;
      entry.style.top = `${top}px`;
    };

    updateButtonPosition();

    const handleResize = () => updateButtonPosition();
    window.addEventListener('resize', handleResize);

    let resizeObserver = null;
    if (typeof ResizeObserver !== 'undefined') {
      resizeObserver = new ResizeObserver(updateButtonPosition);
      resizeObserver.observe(targetDiv);
    }

    teardownPositioning = () => {
      if (resizeObserver) {
        resizeObserver.disconnect();
      }
      window.removeEventListener('resize', handleResize);
    };
  };

  const initMutationObserver = () => {
    const observer = new MutationObserver(() => {
      if (!document.getElementById(ENTRY_ID)) {
        ensureEntryExists();
      }
    });
    observer.observe(document.documentElement, {
      childList: true,
      subtree: true
    });
  };

  const init = async () => {
    await waitForBody();
    ensureEntryExists();
    initMutationObserver();
  };

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init, { once: true });
  } else {
    init();
  }
})();

