const UOM_URL = 'https://api.uom.cn/?mod=cc&act=upload_page';
const DEFAULT_WIDTH = 460;
const DEFAULT_HEIGHT = 640;
let uploadWindowId = null;

const sanitizeBounds = (bounds = {}) => {
  const sanitized = {
    width: Number.isFinite(bounds.width) ? Math.round(bounds.width) : DEFAULT_WIDTH,
    height: Number.isFinite(bounds.height) ? Math.round(bounds.height) : DEFAULT_HEIGHT
  };

  if (Number.isFinite(bounds.left)) {
    sanitized.left = Math.round(bounds.left);
  }
  if (Number.isFinite(bounds.top)) {
    sanitized.top = Math.round(bounds.top);
  }

  return sanitized;
};

const focusExistingWindow = async (bounds) => {
  if (uploadWindowId === null) {
    return false;
  }

  try {
    await chrome.windows.update(uploadWindowId, {
      focused: true,
      ...bounds
    });
    return true;
  } catch (error) {
    console.warn('[UOM 上传助手] 无法复用已有弹窗:', error);
    uploadWindowId = null;
    return false;
  }
};

const createPopupWindow = async (bounds) => {
  const popupWindow = await chrome.windows.create({
    url: UOM_URL,
    type: 'popup',
    focused: true,
    ...bounds
  });
  uploadWindowId = popupWindow?.id ?? null;
};

const handleOpenUploadWindow = async (bounds) => {
  const sanitizedBounds = sanitizeBounds(bounds);
  const reused = await focusExistingWindow(sanitizedBounds);
  if (reused) {
    return { success: true, reused: true };
  }

  await createPopupWindow(sanitizedBounds);
  return { success: true, created: true };
};

chrome.windows.onRemoved.addListener((windowId) => {
  if (uploadWindowId === windowId) {
    uploadWindowId = null;
  }
});

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message?.action !== 'openUploadWindow') {
    return undefined;
  }

  handleOpenUploadWindow(message.payload)
    .then((result) => sendResponse(result))
    .catch((error) => {
      console.error('[UOM 上传助手] 创建弹窗失败:', error);
      sendResponse({ success: false, error: error?.message || '创建弹窗失败' });
    });

  return true;
});


